/*
 * Firebird Open Source JavaEE Connector - JDBC Driver
 *
 * Distributable under LGPL license.
 * You may obtain a copy of the License at http://www.gnu.org/copyleft/lgpl.html
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * LGPL License for more details.
 *
 * This file was created by members of the firebird development team.
 * All individual contributions remain the Copyright (C) of those
 * individuals.  Contributors to this file are either listed here or
 * can be obtained from a source control history command.
 *
 * All rights reserved.
 */
package org.firebirdsql.gds.ng.jna;

import com.sun.jna.Native;
import com.sun.jna.Platform;
import org.firebirdsql.jna.fbclient.FbClientLibrary;
import org.firebirdsql.jna.fbclient.WinFbClientLibrary;
import org.firebirdsql.logging.Logger;
import org.firebirdsql.logging.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
/**
 * Implementation of {@link org.firebirdsql.gds.ng.FbDatabaseFactory} for establishing connection using the
 * Firebird native client library.
 * <p>
 * A separate factory is used for embedded: {@link FbEmbeddedDatabaseFactory}.
 * </p>
 *
 * @author <a href="mailto:mrotteveel@users.sourceforge.net">Mark Rotteveel</a>
 * @since 3.0
 */
public final class FbClientDatabaseFactory extends AbstractNativeDatabaseFactory {

    private static final Logger log = LoggerFactory.getLogger(FbClientDatabaseFactory.class);
    private static final List<String> LIBRARIES_TO_TRY =
            Collections.unmodifiableList(Arrays.asList("fbclient", "fbclient_x64"));
    private static final FbClientDatabaseFactory INSTANCE = new FbClientDatabaseFactory();

    public static FbClientDatabaseFactory getInstance() {
        return INSTANCE;
    }

    @Override
    protected FbClientLibrary createClientLibrary() {
        final List<Throwable> throwables = new ArrayList<>();
        for (String libraryName : LIBRARIES_TO_TRY) {
            try {
                if (Platform.isWindows()) {
                    return Native.loadLibrary(libraryName, WinFbClientLibrary.class);
                } else {
                    return Native.loadLibrary(libraryName, FbClientLibrary.class);
                }
            } catch (RuntimeException | UnsatisfiedLinkError e) {
                throwables.add(e);
                log.debug("Attempt to load " + libraryName + " failed", e);
                // continue with next
            }
        }
        assert throwables.size() == LIBRARIES_TO_TRY.size();
        log.error("Could not load any of the libraries in " + LIBRARIES_TO_TRY + ":");
        for (int idx = 0; idx < LIBRARIES_TO_TRY.size(); idx++) {
            log.error("Loading " + LIBRARIES_TO_TRY.get(idx) + " failed", throwables.get(idx));
        }
        throw new NativeLibraryLoadException("Could not load any of " + LIBRARIES_TO_TRY + "; linking first exception",
                throwables.get(0));
    }

}
